<?php
namespace YES4Trade\Model;

/**
 * Description of category
 *
 * @author mario
 */
class paypal_api extends \YES4Trade\Model\BaseType{
	private const SESS_NAME = 'YES_PPP';
	private const API_URL_PRODUCTION = 'https://api-m.paypal.com';
	private const API_URL_SANDBOX = 'https://api-m.sandbox.paypal.com';

	var $clientId,$secret,$base_url, $sandbox;

	function __construct(){
		$this->sandbox = (MODULE_PAYMENT_YES_PPP_MODE == 'Sandbox') ? true : false;
		$this->base_url = ($this->sandbox === true) ? self::API_URL_SANDBOX : self::API_URL_PRODUCTION;

		// Deine PayPal Client ID und Secret
		$this->clientId = MODULE_PAYMENT_YES_PPP_APICLIENTID;
		$this->secret = MODULE_PAYMENT_YES_PPP_APISECRET;
		if(!isset($_SESSION[self::SESS_NAME])){
			$_SESSION[self::SESS_NAME] = [];
		}
	}

	public function ppcall( string $url, string $accessToken, array $post_fields = [], string $customRequest = '', $use_post = false ){
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->base_url.$url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			"Authorization: Bearer $accessToken",  // Access Token im Header
			"Content-Type: application/json",
			"Accept-Language: de-DE"
		]);
		if(sizeOf($post_fields)){
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post_fields));
		}
		if(!empty($customRequest)){
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $customRequest);			
		}
		if($use_post === true){
			curl_setopt($ch, CURLOPT_POST, true);  // POST-Methode sicherstellen			
		}
		// cURL-Ausführung und Antwort
		$resp = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);  // Statuscode abfragen
		curl_close($ch);
		if(!in_array($httpCode,[200,204,201])){
			throw new \Exception(self::create_error_exception($resp,$httpCode));
		}
		return $resp;
	}

	public static function create_error_exception(string $response,int $httpCode){
		if(empty($response)){
			return 'Paypal Fehler '.$httpCode;
		}
		$json = json_decode($response,true);
		$det = [];
		if(isset($json['details'])){
			foreach($json['details'] as $d){
				$det[] = sprintf('[%s] %s',$d['issue'],$d['description']);
			}
		}
		$string = sprintf('<b>%s (%d)</b><br />%s',$json->message,$httpCode,implode(', ',$det));	
		return $string;
	}

	public function set_paypal_order($paypal_order){
		$_SESSION[self::SESS_NAME]['ORDER'] = $paypal_order;
	}

	public function get_paypal_order(){
		if(!isset($_SESSION[self::SESS_NAME]['ORDER'])){
			return null;
		}
		return $_SESSION[self::SESS_NAME]['ORDER'];
	}

	public function set_access_token( string $token ){
		$_SESSION[self::SESS_NAME]['access_token'] = $token;
	}

	public function get_access_token(){
		if(!isset($_SESSION[self::SESS_NAME]['access_token'])){
			return null;
		}
		return $_SESSION[self::SESS_NAME]['access_token'];
	}

	private static function set_invoice_id(string $value):void{
		$_SESSION[self::SESS_NAME]['invoice_id'] = $value;
	}

	public static function get_invoice_id():string{
		return $_SESSION[self::SESS_NAME]['invoice_id'];
	}


	/**
	 * erzeuge Random Code
	 * verwenden wir als paypal parameter invoice_id um
	 * die order zu identifizieren
	 * @return string
	 */
	private static function create_invoice_id(int $length = 30):string{
		return 'YS-'.substr(bin2hex(random_bytes((int) ceil($length / 2))), 0, $length);
	}

	public function getAccess_token( $admin = false ){
		if($admin){
			global $messageStack;
		}
		// cURL für das Access Token
		$ch = curl_init();
		
		curl_setopt($ch, CURLOPT_URL, $this->base_url."/v1/oauth2/token");
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_POST, true);
		
		// Authentifizierung im Header
		curl_setopt($ch, CURLOPT_USERPWD, sprintf("%s:%s",$this->clientId,$this->secret));
		
		// Formulardaten (Grant Type)
		if(!$admin){
			curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
		}else{
			curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials&scope='.urlencode('https://uri.paypal.com/services/transactions/read').','.urldecode('https://uri.paypal.com/services/reporting/read'));
		}
		
		// cURL-Ausführung und Antwort
		$response = curl_exec($ch);
		curl_close($ch);
		// Antwort dekodieren
		$data = json_decode($response, true);
		if(isset($data['error'])){
			if(!$admin){
				xtc_redirect(xtc_href_link(FILENAME_SHOPPING_CART,'error_message='.urlencode($data['error_description'])));
			}else{
				$messageStack->add_session('Cant get Paypal AuthToken. '.$data['error'].': '.$data['error_description']);
				xtc_redirect(xtc_href_link(FILENAME_START));
			}
		}
		$this->set_access_token($data['access_token']);
		return $data['access_token'];
	}

	public function createOrder($order, $order_totals = [], $express_checkout = true){
		$invoice_id = self::create_invoice_id(); // eindeutiger schluessel
		self::set_invoice_id($invoice_id); // in session speichern
		$total = self::get_order_total( $order,$express_checkout,$order_totals ); // warenkorb, nur zwischensumme
		$orderData = [
			'intent' => 'CAPTURE',
			'purchase_units' => [
				[
					'amount' => [
						'currency_code' => $order->Price->actualCurr,
						'value' => "$total"
					],
					'invoice_id'=>self::get_invoice_id()
				]
			],
			'application_context' => [
				'return_url' => ($express_checkout)?xtc_href_link(FILENAME_CHECKOUT_SHIPPING):xtc_href_link(FILENAME_CHECKOUT_PROCESS),
				'cancel_url' => xtc_href_link(FILENAME_SHOPPING_CART),
				'shipping_preference'=>'GET_FROM_FILE',
			]
		];
		$response = $this->ppcall(
			"/v2/checkout/orders",
			$this->get_access_token(),
			$orderData
		);
		$paypal_order = json_decode($response, true);
		if(is_countable($paypal_order) and $paypal_order['status'] === 'CREATED'){
			$this->set_paypal_order($paypal_order);
			foreach ($paypal_order['links'] as $link) {
				if ($link['rel'] === 'approve') {
					header("Location: " . $link['href']);
					exit;
				}
			}
		}
		return $response;
	}

	public function updateOrderShippingAdress(string $orderId, array $post_data){
		$patchData = [[
			'op' => 'replace',
			'path' => "/purchase_units/@reference_id=='default'/shipping/address",
			'value' => [
				"address_line_1"=> $post_data['street_address'],
				"address_line_2" => $post_data['suburb']??'',
				"admin_area_2"=> $post_data['city'],
				"admin_area_1" => $post_data['state']??'', // STATE
				'organization'=>$post_data['company']??'',
				"postal_code"=> $post_data['postcode'],
				"country_code"=> self::yes_get_iso_code2_from_countries_id($post_data['country']),
	  		]
		],[
			'op' => 'replace',
			'path' => "/purchase_units/@reference_id=='default'/shipping/name",
			'value' => [
				'full_name'=>$post_data['firstname'].' '.$post_data['lastname'],  // Vollständiger Name (optional),
				'first_name'=>$post_data['firstname'],
				'last_name'=>$post_data['lastname'],
			]
		]];
		$response = $this->ppcall("/v2/checkout/orders/".$orderId, 
			$this->get_access_token(),
			$patchData, 
			'PATCH'
		);
		if(empty($response)){
			return true;
		}
		return false;
	}

	public function updateOrderBillingAdress(string $orderId, array $post_data){
		$patchData = [[
			'op' => 'replace',
			'path' => "/purchase_units/@reference_id=='default'/shipping/address",
			'value' => [
				"address_line_1"=> $post_data['street_address'],
				"address_line_2" => $post_data['suburb']??'',
				"admin_area_2"=> $post_data['city'],
				"admin_area_1" => $post_data['state']??'', // STATE
				'organization'=>$post_data['company']??'',
				"postal_code"=> $post_data['postcode'],
				"country_code"=> self::yes_get_iso_code2_from_countries_id($post_data['country']),
	  		]
		],[
			'op' => 'replace',
			'path' => "/purchase_units/@reference_id=='default'/shipping/name",
			'value' => [
				'full_name'=>$post_data['firstname'].' '.$post_data['lastname'],  // Vollständiger Name (optional),
				'first_name'=>$post_data['firstname'],
				'last_name'=>$post_data['lastname'],
			]
		]];
		$response = $this->ppcall("/v2/checkout/orders/".$orderId, 
			$this->get_access_token(),
			$patchData, 
			'PATCH'
		);
		return true;
	}

	private static function get_order_total( $order, $from_cart = false, $order_totals = []){
		$default_shipping_amount = 0;
		if($from_cart === true){
			if(isset($_SESSION['default_shipping_countries_id'])){
				// alt
				//$default_shipping_countries_id = $_POST['default_shipping_countries_id'];
				$default_shipping_countries_id = $_SESSION['default_shipping_countries_id'];
			}else{
				$default_shipping_countries_id = yes_get_store_country_id();
			}
			$temporary_shippingcosts = $_SESSION['cart']->get_temporary_shippingcosts($default_shipping_countries_id);
            if(isset($temporary_shippingcosts['QUOTES'])){
                foreach($temporary_shippingcosts['QUOTES'] as $ts){
                	if($ts['id'] == 'selfpickup'){
                		continue;
                	}
                        foreach($ts['methods'] as $tsm){
                            $default_shipping_amount = floatval($tsm['price_raw']);
                        }
                }
            }
			$keys = ['ot_subtotal','ot_subtotal_no_tax'];
		}else{
			$keys = ['ot_total'];
		}
		foreach($order_totals as $ot){
			if(in_array($ot['code'], $keys)){
				$ot_val = floatval($ot['value']);
				if($ot['value'] > 0){
					return xtc_round($ot_val+$default_shipping_amount,2);
				}
			}
		}
		return 0;
	}

	public function updateOrder($order, $order_totals){ // in der process_shipping
		$total = self::get_order_total( $order, false, $order_totals );
		$patchData = [[
			'op' => 'replace',
			'path' => "/purchase_units/@reference_id=='default'/amount",
			'value' => [
				'currency_code' => $order->Price->actualCurr,
				'value' => "$total",
				/*'breakdown' => [
					'item_total' => ['currency_code' => 'EUR', 'value' => '49.99'],
					'shipping'   => ['currency_code' => 'EUR', 'value' => '5.00']
				]*/
			]
		]];
		$response = $this->ppcall("/v2/checkout/orders/".$this->getOrderToken(), $this->get_access_token(),$patchData, 'PATCH');
	}

	public function capture_from_order(){
		global $order;
		$data = [
			"intent" => "CAPTURE",
			"purchase_units" => [
				[
					//"reference_id" => $orders_id,  // Deine Bestell-ID
					"amount" => [
						"currency_code" => $order->info['currency'],
						"value" => xtc_round($order->info['total'],2)
					]
				]
			],
			"application_context" => [
				"return_url" => xtc_href_link(FILENAME_CHECKOUT_PROCESS),  // Die URL, auf die der Kunde nach Abschluss der Zahlung weitergeleitet wird
				"cancel_url" => xtc_href_link(FILENAME_CHECKOUT_SHIPPING)// URL, wenn der Kunde die Zahlung abbricht
			]
		];
		$response = $this->ppcall('/v2/checkout/orders',$this->get_access_token(),$data);
		$resp = json_decode($response);
		$this->set_transaction_id($resp->id);
		foreach ($resp->links as $link) {
			if ($link->rel == 'approve') {
				header('Location: ' . $link->href);  // Der Kunde wird zur PayPal-Website weitergeleitet
				exit();
			}
		}
	}

	public function capture(string $token){
		$use_post = true;
		$post_params = [];
		$custom_request = '';
		$response = $this->ppcall(sprintf("/v2/checkout/orders/%s/capture",
				$token
			), $this->get_access_token(), $post_params, $custom_request, $use_post
		);
		
		$result = json_decode($response, true);
		
		if ($result['status'] === 'COMPLETED') {
			$transactionId = $result['purchase_units'][0]['payments']['captures'][0]['id'];
			$this->set_transaction_id($transactionId);
		}
	}

	public function set_transaction_id( string $transaction_id ){
		$_SESSION[self::SESS_NAME]['transaction_id'] = $transaction_id;
	}

	public function get_transaction_id(){
		if(!isset($_SESSION[self::SESS_NAME]['transaction_id'])){
			return null;
		}
		return $_SESSION[self::SESS_NAME]['transaction_id'];
	}

	public function saveOrderToken(string $token){
		$_SESSION[self::SESS_NAME]['order_token'] = $token;
	}

	public function getOrderToken(){
		return $_SESSION[self::SESS_NAME]['order_token']??null;
	}

	public function savePayerID(string $payerID){
		$_SESSION[self::SESS_NAME]['payerID'] = $payerID;
	}

	public function getPayerID(){
		return $_SESSION[self::SESS_NAME]['payerID']??null;
	}

	public function getOrder(string $order_token){
		$response = $this->ppcall('/v2/checkout/orders/'.$order_token, $this->get_access_token());
		return json_decode($response,true);
	}

	public static function get_yes_orders_customer_array(array $paypal_order){
		if(isset($_SESSION[self::SESS_NAME]['yes_billing_address'])){
			$email_address = $_SESSION[self::SESS_NAME]['yes_billing_address']['email_address'];
			if(empty($email_address)){
				if(isset($paypal_order['payment_source']['paypal']['email_address'])){
					$email_address = $paypal_order['payment_source']['paypal']['email_address'];
				}
			}
			return [
				'firstname' => $_SESSION[self::SESS_NAME]['yes_billing_address']['firstname'],
				'lastname' => $_SESSION[self::SESS_NAME]['yes_billing_address']['lastname'],
				'company'=>$_SESSION[self::SESS_NAME]['yes_billing_address']['company'],
				'street_address' => $_SESSION[self::SESS_NAME]['yes_billing_address']['street_address'],
				'suburb' => $_SESSION[self::SESS_NAME]['yes_billing_address']['suburb']??'',
				'city' => $_SESSION[self::SESS_NAME]['yes_billing_address']['city'],
				'postcode' => $_SESSION[self::SESS_NAME]['yes_billing_address']['postcode'],
				'country' => [
					'id' => $_SESSION[self::SESS_NAME]['yes_billing_address']['country']['id'],
					'title' => $_SESSION[self::SESS_NAME]['yes_billing_address']['country']['title'],
					'iso_code_2' => $_SESSION[self::SESS_NAME]['yes_billing_address']['country']['iso_code_2'],
					'iso_code_3' => $_SESSION[self::SESS_NAME]['yes_billing_address']['country']['iso_code_3'],
				],
				'country_id'=>$_SESSION[self::SESS_NAME]['yes_billing_address']['country']['id'],
				'zone_id'=>0,
				'format_id' => 5,
				'email_address' => $email_address
			];
	
		}else{
			$pu = current($paypal_order['purchase_units']);
			$address = $pu['shipping'];
			$names = self::splitName($address['name']['full_name']);
			$country_id = yes_get_country_id_from_iso2($address['address']['country_code']);
			$country_name = xtc_get_country_name($country_id);
		}
		return [
			'firstname' => $names['vorname'],
			'lastname' => $names['nachname'],
			'company'=>$address['name']['organization'],
			'street_address' => $address['address']['address_line_1'],
			'suburb' => $address['address']['address_line_2']??'',
			'city' => $address['address']['admin_area_2'],
			'postcode' => $address['address']['postal_code'],
			'country' => [
				'id' => $country_id,
				'title' => $country_name,
				'iso_code_2' => $address['address']['country_code'],
				'iso_code_3' => self::yes_get_iso_code3($country_name),
			],
			'country_id'=>$country_id,
			'zone_id'=>0,
			'format_id' => 5,
			'email_address' => $paypal_order['payment_source']['paypal']['email_address']
		];
	}

	public static function get_yes_orders_delivery_array(array $paypal_order){
		$pu = current($paypal_order['purchase_units']);
		$address = $pu['shipping'];
		$names = self::splitName($address['name']['full_name']);
		if(!function_exists('yes_get_country_id_from_iso2')){
			require_once(DIR_FS_INC.'yes_get_country_id_from_iso2.inc.php');
		}
		$country_id = yes_get_country_id_from_iso2($address['address']['country_code']);
		$country_name = xtc_get_country_name($country_id);
		return [
			'firstname' => $names['vorname'],
			'lastname' => $names['nachname'],
			'company'=>$address['name']['organization'],
			'street_address' => $address['address']['address_line_1'],
			'suburb' => $address['address']['address_line_2']??'',
			'city' => $address['address']['admin_area_2'],
			'postcode' => $address['address']['postal_code'],
			'country' => [
				'id' => $country_id,
				'title' => $country_name,
				'iso_code_2' => $address['address']['country_code'],
				'iso_code_3' => self::yes_get_iso_code3($country_name),
			],
			'country_id'=>$country_id,
			'zone_id'=>0,
			'format_id' => 5
		];
	}

	public static function splitName(string $fullName): array {
		$parts = preg_split('/\s+/', trim($fullName));
		
		if (count($parts) === 0) {
			return ['vorname' => '', 'nachname' => ''];
		}
	
		$nachname = array_pop($parts);
		$vorname = implode(' ', $parts);
	
		return ['vorname' => $vorname, 'nachname' => $nachname];
	}
	
	public static function yes_get_iso_code3($country_name){
		$query = yes_db_query(sprintf(
			"SELECT countries_iso_code_3 FROM countries WHERE countries_name='%s'",
			$country_name
		),true);
		$record = current($query);
		return $record['countries_iso_code_3'];
	}
	
	public static function yes_get_iso_code2_from_countries_id(int $countries_id){
		$query = yes_db_query(sprintf(
			"SELECT countries_iso_code_2 FROM countries WHERE countries_id='%d'",
			$countries_id
		),true);
		$record = current($query);
		return $record['countries_iso_code_2'];
	}
	public static function get_express_button(int $default_shipping_countries_id, float $shipping_costs, string $currency){
		return sprintf('<a class="paypal_express_button" href="%s">Paypal Express</a><div class="paypal_express_info">&#128666; %s: %s %s</div>',
			xtc_href_link(FILENAME_CHECKOUT_SHIPPING,'action=paypal_express_create_order'),
			xtc_get_country_name($default_shipping_countries_id),
			number_format($shipping_costs,2,',','.'),
			$currency
		);
	}

	public static function clear_session(){
		$_SESSION[self::SESS_NAME] = null;
		unset($_SESSION[self::SESS_NAME]);
	}

	public function getTransactions(\DateTime $startDate, \DateTime $endDate, int $page = 1){
		$access_token = $this->get_access_token();
		if($access_token === null){
			$access_token = $this->getAccess_token(true); // admin
			$this->set_access_token($access_token);
		}
		return $this->ppcall(
			sprintf("/v1/reporting/transactions?start_date=%s&end_date=%s&page_size=100&page=%d",
				$startDate->format('Y-m-d\TH:i:s\Z'),
				$endDate->format('Y-m-d\TH:i:s\Z'),
				$page,
			),
			$access_token
		);
	}

	/**
	 * gibt die txn_id aus tabelle paypal zurueck, wenn xtc_order_id = orders_id
	 * leer wenn keine zuweisung
	 * @param int $orders_id
	 * 
	 * @return string
	 */
	public static function getOrderAllocation(int $orders_id){
		$query = yes_query("SELECT txn_id FROM paypal WHERE xtc_order_id=:id",['id'=>$orders_id],true);
		if($query !== null){
			return $query['txn_id'];
		}
		return '';
	}

	/**
	 * holt aus tabelle paypal den xtc_order_id eintrag zu einer txn_id
	 * gibt null zurueck wenn kein eintrag
	 * 
	 * @param string $paypal_order_id
	 * 
	 * @return int|null
	 */
	public static function get_yes_orders_id_from_paypal_order_id( string $paypal_order_id ):int|null{
		$query = yes_query(
			"SELECT xtc_order_id FROM paypal WHERE txn_id=:id",[
				'id'=>$paypal_order_id
			],true
		);
		if($query === false){
			return null;
		}
		return intval($query['xtc_order_id']);
	}

	public function execute(string $paymentId, string $PayerId){
		$paypal_url = '/v2/payments/payment/' . $paymentId . '/execute';
		$accessToken = $this->get_access_token();
		$executeData = [
			'payer_id' => $PayerId,  // PayerID aus der Rueckkehr-URL
		];
		$response = $this->ppcall($paypal_url, $accessToken, $executeData,'',true);
		$responseData = json_decode($response, true);
		if ($responseData['state'] == 'approved') {
			return true;
			// Hier kannst du den Status der Bestellung auf "bezahlt" setzen oder etwas anderes machen
		}
		throw new \Exception('Paypal Zahlung fehlgeschlagen');
	}

	function getSale(string $transaction_id){
		return $this->ppcall('/v2/payments/sale/'.$transaction_id,
			$this->get_access_token()
		);
	}

	public function getReport(\DateTime $startDate, \DateTime $endDate, string $transaction_id=''){
		if(!empty($transaction_id)){
			return $this->ppcall(
				'/v1/reporting/transactions?transaction_id='.$transaction_id.'&start_date='.urlencode($startDate->format('Y-m-d\TH:i:s\Z')).'&end_date='.urlencode($endDate->format('Y-m-d\TH:i:s\Z')).'&fields=all',
				$this->get_access_token()
			);
		}
		return $this->ppcall('/v1/reporting/transactions?start_date='.urlencode($startDate->format('Y-m-d\TH:i:s\Z')).'&end_date='.urlencode($endDate->format('Y-m-d\TH:i:s\Z')).'&fields=all', $this->get_access_token());
	}

	public function verifyWebhook(array $headers, string $rawBody, string $webhookId){
		$accessToken = $this->get_access_token();
		if($accessToken === null){
			$accessToken = $this->getAccess_token();
			$this->set_access_token($accessToken);
		}
		$jbody = json_decode($rawBody);
		//yes_mail_debug("Header:".print_r($headers,true)."\n\nBody (".gettype($jbody)."):".print_r($jbody,true)."\n\nwebhookId:".$webhookId."\n\njson_decode error: ".json_last_error_msg());
		$data = [
			'auth_algo'         => $headers['PAYPAL-AUTH-ALGO'] ?? '',
			'cert_url'          => $headers['PAYPAL-CERT-URL'] ?? '',
			'transmission_id'   => $headers['PAYPAL-TRANSMISSION-ID'] ?? '',
			'transmission_sig'  => $headers['PAYPAL-TRANSMISSION-SIG'] ?? '',
			'transmission_time' => $headers['PAYPAL-TRANSMISSION-TIME'] ?? '',
			'webhook_id'        => trim("$webhookId"),
			'webhook_event'     => json_decode($rawBody),
		];
		$response = $this->ppcall(
			'/v1/notifications/verify-webhook-signature',
			$this->get_access_token(),$data,'',true
		);
		$result = json_decode($response, true);
		return isset($result['verification_status']) && $result['verification_status'] === 'SUCCESS';
	}

	public function get_webhook_id(){
		$file = \main::get_secure_path().'YES_PPP_WEBHOOK_ID.'.$this->clientId;
		if(!is_file($file)){
			yes_mail_debug('cant find webhook under '.$file,false,'cant find webhook');
			return null;
		}
		return trim(file_get_contents($file));
	}

	public function save_webhook_id(string $webhook_id){
		$file = \main::get_secure_path().'YES_PPP_WEBHOOK_ID.'.$this->clientId;
		return file_put_contents($file, $webhook_id);
	}

	public function searchTransactions(array $params){
		$accessToken = $this->get_access_token();
		if($accessToken === null){
			$accessToken = $this->getAccess_token();
			$this->set_access_token($accessToken);
		}
		$startDate = (isset($_GET['start_date'])) ? (new \DateTime($_GET['start_date']))->format(DATE_ATOM) : (new \DateTime('-2 Weeks'))->format(DATE_ATOM);
		$endDate = (isset($_GET['end_date'])) ? (new \DateTime($_GET['end_date']))->format(DATE_ATOM) : (new \DateTime())->format(DATE_ATOM);
		
		$filter_params = [
			'start_date'=>$startDate,
			'end_date'=>$endDate,
			'fields'=>'all'
		];
		foreach($params as $k=>$v){
			switch($k){
				case 'transaction_id':
					$filter_params['transaction_id'] = $v;
					break;
			}
		}
		$response = $this->ppcall(
			'/v1/reporting/transactions/?'.http_build_query($filter_params),
			$this->get_access_token()
		);
		return json_decode($response, true);
	}


	/**
	 * Hole xtc_order_id aus Tabelle Paypal
     * Aus der Tabelle paypal wird anhand der invoice_id (Spalte invoice) abgeglichen
	 * ob wir eine passende xtc_order_id haben
	 * gibt 0 zurück wenn nichts gefunden wurde, ansonsten die xtc_order_id
	 * 
	 * @param string $invoice_id
	 * 
	 * @return int
	 */
	public static function get_yes_orders_id_from_invoice_id(string $invoice_id):int{
		$query = yes_query("SELECT xtc_order_id FROM paypal WHERE invoice=:invoice",[
			'invoice'=>$invoice_id
		],true);
		if($query !== false){
			return intval($query['xtc_order_id']);
		}
		return 0;
	}

	public static function get_ip_id(string $transaction_id):int{
		$query = yes_query("SELECT id FROM incoming_payments WHERE verwendungszweck3=:txn",[
			'txn'=>'TACode: '.$transaction_id
		],true);
		return intval($query['id']??0);
	}

	public function refund(string $capture_id, float $amount, string $note = ''){
		$accessToken = $this->get_access_token();
		if($accessToken === null){
			$accessToken = $this->getAccess_token();
			$this->set_access_token($accessToken);
		}
		$params = [
			'amount'=>[
				'value'=>"".number_format($amount,2,'.',''),
				'currency_code'=>constant('DEFAULT_CURRENCY')
			],
			'note_to_payer'=>(!empty($note))?$note:'Refund '.constant('STORE_NAME'),
		];
		$response = $this->ppcall( // post
			'/v2/payments/captures/'.$capture_id.'/refund',
			$this->get_access_token(),$params,'',true
		);
		return json_decode($response, true);
	}
}

